/* $Id: arbsrct2.c,v 1.2 1998/09/16 02:04:57 ericb Exp $ */
/* Copyright (C) 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Test for obscure arb source firmware error found by production test */

#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "e1432.h"

#define	NMOD_MAX	4
#define	NCHAN_MAX	(NMOD_MAX * E1432_CHANS)

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: arbsrct2.c,v 1.2 1998/09/16 02:04:57 ericb Exp $";
static const char *progname;

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, SHORTSIZ16 *chan_list)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, nc;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    for (i = 0; i < nmod; i++)
	nc += hwconfig[i].source_chans;
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_SOURCE_CHAN(i + 1);

    *group = e1432_create_channel_group(*hw, nc, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    return 0;
}

/*ARGSUSED*/
static int
setup(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
      int src_mode, int sb_mode)
{
    int     loop;

    for (loop = 0; loop < 2; loop++)
    {
	CHECK(e1432_set_active(hw, group, E1432_CHANNEL_ON));
	CHECK(e1432_set_source_mode(hw, group, src_mode));
	CHECK(e1432_set_srcbuffer_mode(hw, group, sb_mode));
	CHECK(e1432_set_srcbuffer_size(hw, group, 20480));
	/* For the original bug, the second time through the loop this
	   srcbuffer_init function would return an error. */
	CHECK(e1432_set_srcbuffer_init(hw, group, E1432_SRCBUFFER_INIT_EMPTY));
	CHECK(e1432_preset(hw, group));
    }
    return 0;
}

/*ARGSUSED*/
static int
run(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list)
{
    CHECK(e1432_init_measure(hw, group));
    CHECK(e1432_finish_measure(hw, group));
    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-uV] [-L laddr]\n"
		   "\t    [-m src_mode] [-n nchan] [-N nmod]\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-m: Set source mode to <src_mode>, default ARB\n"
		   "\t-n: Use <nchan> channels, default all\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-u: Print this usage message\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX];
    char   *p;
    int     i, c, nmod, src_mode, group, nchan;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    laddr[0] = 8;
    src_mode = E1432_SOURCE_MODE_ARB;
    nchan = -1;			/* Meaning use all channels */
    nmod = 1;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "L:m:n:N:uV")) != -1)
	switch (c)
	{
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'm':
	    p = optarg;
	    while (*p != '\0')
	    {
		*p = tolower(*p);
		p++;
	    }
	    if (strcmp(optarg, "arb") == 0)
		src_mode = E1432_SOURCE_MODE_ARB;
	    else if (strcmp(optarg, "barb") == 0)
		src_mode = E1432_SOURCE_MODE_BARB;
	    else
	    {
		(void) fprintf(stderr,
			       "%s: invalid source mode: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'n':
	    nchan = strtol(optarg, &p, 0);
	    if (optarg == p || nchan < -1 || nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    if (init(nmod, laddr, &hw, &group, &nchan, chan_list) < 0)
	return EXIT_FAILURE;
    if (setup(hw, group, nchan, chan_list, src_mode,
	      E1432_SRCBUFFER_PERIODIC_A) < 0)
	return EXIT_FAILURE;
    if (setup(hw, group, nchan, chan_list, src_mode,
	      E1432_SRCBUFFER_PERIODIC_AB) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
